"use client";
import { FormEvent, useState } from "react";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";

export default function ForgotPasswordPage() {
  const { t } = useLanguage();

  const [email, setEmail] = useState("");
  const [err, setErr] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState(false);
  const [accountExists, setAccountExists] = useState<boolean | null>(null);

  const onSubmit = async (e: FormEvent) => {
    e.preventDefault();
    setErr(null);
    setLoading(true);
    setAccountExists(null);

    try {
      const res = await fetch("/api/auth/forgot-password", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ email }),
      });

      const json = await res.json();

      if (!json.ok) {
        throw new Error(json.error || "Failed to send reset email");
      }

      setAccountExists(json.accountExists);
      setSuccess(true);
    } catch (error: any) {
      setErr(error.message || "Failed to send reset email");
    } finally {
      setLoading(false);
    }
  };

  if (success) {
    if (accountExists === false) {
      // Account doesn't exist - show message to create account
      return (
        <div className='min-h-screen bg-sidebar flex items-center justify-center px-4'>
          <div className='w-full max-w-md rounded-2xl border border-card/60 bg-card shadow-xl p-8 text-center'>
            <div className='w-16 h-16 bg-yellow-100 rounded-full flex items-center justify-center mx-auto mb-4'>
              <svg
                className='w-8 h-8 text-yellow-600'
                fill='none'
                stroke='currentColor'
                viewBox='0 0 24 24'
              >
                <path
                  strokeLinecap='round'
                  strokeLinejoin='round'
                  strokeWidth={2}
                  d='M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z'
                />
              </svg>
            </div>
            <h1 className='text-2xl font-bold text-foreground mb-2'>
              Account Not Found
            </h1>
            <p className='text-muted mb-6'>
              No account found with this email address. Please create an account first.
            </p>
            <div className='flex flex-col sm:flex-row gap-3'>
              <Link
                href='/signup'
                className='flex-1 inline-block rounded-xl bg-primary px-4 py-2.5 text-white font-medium hover:opacity-90 transition text-center'
              >
                Create Account
              </Link>
              <Link
                href='/login'
                className='flex-1 inline-block rounded-xl bg-gray-200 px-4 py-2.5 text-gray-800 font-medium hover:bg-gray-300 transition text-center'
              >
                Back to Login
              </Link>
            </div>
          </div>
        </div>
      );
    }

    // Account exists - show success message
    return (
      <div className='min-h-screen bg-sidebar flex items-center justify-center px-4'>
        <div className='w-full max-w-md rounded-2xl border border-card/60 bg-card shadow-xl p-8 text-center'>
          <div className='w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4'>
            <svg
              className='w-8 h-8 text-green-600'
              fill='none'
              stroke='currentColor'
              viewBox='0 0 24 24'
            >
              <path
                strokeLinecap='round'
                strokeLinejoin='round'
                strokeWidth={2}
                d='M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z'
              />
            </svg>
          </div>
          <h1 className='text-2xl font-bold text-foreground mb-2'>
            Check Your Email
          </h1>
          <p className='text-muted mb-6'>
            We've sent you a password reset link. Please check your inbox and follow the instructions.
          </p>
          <Link
            href='/login'
            className='inline-block w-full rounded-xl bg-primary px-4 py-2.5 text-white font-medium hover:opacity-90 transition'
          >
            Back to Login
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className='min-h-screen bg-sidebar relative flex items-center justify-center px-4'>
      {/* Background decoration */}
      <div
        aria-hidden
        className='pointer-events-none absolute inset-0 -z-10 overflow-hidden'
      >
        <div className='absolute -top-32 -right-24 w-80 h-80 bg-sidebar-accent/40 rounded-full blur-3xl' />
        <div className='absolute -bottom-32 -left-24 w-80 h-80 bg-primary/20 rounded-full blur-3xl' />
      </div>

      <div className='w-full max-w-md'>
        <div className='rounded-2xl border border-card/60 bg-card shadow-xl'>
          {/* Header */}
          <div className='px-8 pt-8 pb-5 text-center'>
            <div className='mx-auto mb-4 flex h-16 w-16 items-center justify-center rounded-full bg-primary/10'>
              <svg
                className='h-8 w-8 text-primary'
                fill='none'
                stroke='currentColor'
                viewBox='0 0 24 24'
              >
                <path
                  strokeLinecap='round'
                  strokeLinejoin='round'
                  strokeWidth={2}
                  d='M15 7a2 2 0 012 2m4 0a6 6 0 01-7.743 5.743L11 17H9v2H7v2H4a1 1 0 01-1-1v-2.586a1 1 0 01.293-.707l5.964-5.964A6 6 0 1121 9z'
                />
              </svg>
            </div>
            <h1 className='text-3xl font-bold text-foreground mb-1'>
              Forgot Password?
            </h1>
            <p className='text-muted'>
              Enter your email address and we'll send you a link to reset your password.
            </p>
          </div>

          {/* Form */}
          <form onSubmit={onSubmit} className='px-8 pb-8 space-y-6'>
            {err && (
              <div
                role='alert'
                aria-live='polite'
                className='bg-red-50 border border-red-200 rounded-lg p-4 flex items-start text-sm text-red-800'
              >
                <svg
                  className='w-4 h-4 mr-2 mt-0.5'
                  fill='none'
                  stroke='currentColor'
                  viewBox='0 0 24 24'
                >
                  <path
                    strokeLinecap='round'
                    strokeLinejoin='round'
                    strokeWidth={2}
                    d='M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z'
                  />
                </svg>
                {err}
              </div>
            )}

            <div>
              <label
                htmlFor='email'
                className='block text-sm font-medium text-foreground mb-2'
              >
                Email Address
              </label>
              <input
                id='email'
                type='email'
                required
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                className='w-full rounded-xl border border-input bg-background px-3.5 py-2.5 text-foreground placeholder:text-muted focus:outline-none focus:ring-2 focus:ring-primary/50'
                placeholder='Enter your email address'
                autoComplete='email'
              />
            </div>

            <button
              type='submit'
              disabled={loading}
              className='w-full inline-flex items-center justify-center rounded-xl bg-primary px-4 py-2.5 text-white font-medium shadow hover:opacity-90 disabled:opacity-50 disabled:cursor-not-allowed transition'
            >
              {loading ? (
                <>
                  <svg
                    className='animate-spin -ml-1 mr-2 h-5 w-5'
                    xmlns='http://www.w3.org/2000/svg'
                    fill='none'
                    viewBox='0 0 24 24'
                  >
                    <circle
                      className='opacity-25'
                      cx='12'
                      cy='12'
                      r='10'
                      stroke='currentColor'
                      strokeWidth='4'
                    ></circle>
                    <path
                      className='opacity-75'
                      fill='currentColor'
                      d='M4 12a8 8 0 018-8V1C5.373 1 1 5.373 1 12h3z'
                    ></path>
                  </svg>
                  Sending...
                </>
              ) : (
                "Send Reset Link"
              )}
            </button>
          </form>

          {/* Footer Links */}
          <div className='px-8 pb-6 pt-2 border-t border-card text-center'>
            <p className='text-sm'>
              <Link
                href='/login'
                className='text-muted hover:text-foreground inline-flex items-center justify-center'
              >
                <svg
                  className='w-4 h-4 mr-1'
                  fill='none'
                  stroke='currentColor'
                  viewBox='0 0 24 24'
                >
                  <path
                    strokeLinecap='round'
                    strokeLinejoin='round'
                    strokeWidth={2}
                    d='M10 19l-7-7m0 0l7-7m-7 7h18'
                  />
                </svg>
                Back to Login
              </Link>
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}

